#!/usr/bin/env python3 
# coding: utf-8

# 
# Import Libraries
#

import nest
import numpy

import argparse

nest.set_verbosity(100)

parser = argparse.ArgumentParser(description='STDP test with Tsodyks Synapse')
parser.add_argument('--A2', metavar='float', type=float, help='Magnitude of the second current injectin.', default=500.0)

arg = parser.parse_args()

# Best Parameter

A_in = 5.5
Tau_rec_in = 35.0
Tau_fac_in = 1000.0

# Setting up the parameters
# Copy from nest-simulator homepage

h = 0.1                # simulation step size (ms)
Tau = 15.              # membrane time constant
Theta = -50.           # threshold
E_L = -65.             # reset potential of membrane potential
R = 1                  # 100 M Ohm
C = Tau / R            # Tau (ms)/R in NEST units
TauR = 2.              # refractory time
                       
Tau_psc = 2.           # time constant of PSC (= Tau_inact)
Tau_rec = Tau_rec_in   # recovery time
Tau_fac = Tau_fac_in   # facilitation time
A = A_in               # PSC weight in pA

neuron_param = {
#               "tau_m": Tau,
                "t_ref": TauR,
                "tau_syn_ex": Tau_psc,
                "tau_syn_in": Tau_psc,
                "C_m": C,
                "V_reset": E_L,
                "E_L": E_L,
                "V_m": E_L,
                "V_th": Theta
                }

syn_param = {
             "tau_psc": Tau_psc,
             "tau_rec": Tau_rec,
             "tau_fac": Tau_fac,
             "delay": 0.1,
             "weight": A,
             "u": 0.0,
             "x": 1.0
             , "Wmax": A * 2.0
             , "mu_plus": 1.0
             , "mu_minus": 1.0
             , "lambda": 0.1
             }

gamma_param = [
        [1.40, 26.5],
        [3.12, 10.1],
        [2.94, 12.6]
        ]
 
for delta_t in numpy.arange(-20,20.01,0.5): 

    print(arg.A2, delta_t, end="\t")

    for alpha,beta in gamma_param:

        start_sample = list()
        end_sample = list()

        for sample_i in range(1000):
            
            # Reset the simulator
            
            nest.ResetKernel()
            nest.SetKernelStatus({"resolution": h})
            
            # Create objects in the system
            
            dc_gen1  = nest.Create("dc_generator", 10)
            dc_gen2  = nest.Create("dc_generator", 10)
            neurons = nest.Create("iaf_cond_exp", 2)
            volts0  = nest.Create('multimeter')
            volts1  = nest.Create('multimeter')
            
            # Initial the object
            
            nest.SetStatus(neurons, neuron_param)
            nest.SetStatus(volts0, {"label": "voltmeter", "withtime": True, "withgid": True, 
                                    "interval": 0.1,'record_from': ['V_m', 'g_ex']})
            nest.SetStatus(volts1, {"label": "voltmeter", "withtime": True, "withgid": True, 
                                    "interval": 0.1,'record_from': ['V_m', 'g_ex']})
            
            act_time_1 = list()
            act_time_2 = list()
            
            for idx in range(10):
            
                start_time = 50.0 + 50.0 * idx
                end_time   = start_time + 1.0
                act_time_1.append({"amplitude": 500.0, "start": start_time, "stop": end_time})
            
                start_time = 50.0 + 50.0 * idx + delta_t
                end_time   = start_time + 1.0
                act_time_2.append({"amplitude": arg.A2, "start": start_time, "stop": end_time})
                
            nest.SetStatus(dc_gen1, act_time_1)
            nest.SetStatus(dc_gen2, act_time_2)
            
            # Connect the sp_gen to neuron #0
            
            nest.Connect(dc_gen1, [neurons[0]])
            nest.Connect(dc_gen2, [neurons[1]])
            
            # Connect two neurons with a STDP-Tsodyks synapse
            
            for idx in range(1):
                while True:
                    U_local = numpy.random.gamma(alpha, beta) / 140.0
                    if U_local < 1.0:
                        break
                syn_param_tmp = syn_param.copy()
                syn_param_tmp["U"] = U_local
                syn_param_tmp["weight"] = syn_param["weight"] 
                syn_param_tmp["Wmax"] = syn_param["Wmax"] 
                syn_param_tmp.update({"model": "stdp_tsodyks_synapse"})
                nest.Connect([neurons[0]], [neurons[1]], syn_spec=syn_param_tmp)
            
#            nest.CopyModel("stdp_goda_tsodyks_synapse", "stdp_dyn_syn", syn_param)
#            nest.Connect([neurons[0]], [neurons[1]], syn_spec="stdp_dyn_syn")
            
            # Connect voltmeters to neurons
            
            nest.Connect(volts0, [neurons[0]])
            nest.Connect(volts1, [neurons[1]])
            
            # Run the simulation
            
            dt = 1.0
            t_start = 0.0
            t_end = numpy.max([act_time_1[-1]["stop"], act_time_2[-1]["stop"]]) + 50.0
            
            tmp_stdp_syn = numpy.sum(nest.GetStatus(nest.GetConnections([neurons[0]], [neurons[1]]), "weight"))

            #print(tmp_stdp_syn[0], end="\t")

            start_sample.append(tmp_stdp_syn)
            
            for t in numpy.arange(t_start,t_end+0.5*delta_t,dt):
                #tmp_stdp_syn = nest.GetStatus(nest.GetConnections([neurons[0]], [neurons[1]], "stdp_dyn_syn"), "weight")
                #print(tmp_stdp_syn[0], end="\n")
            
                nest.Simulate(dt)
            
            tmp_stdp_syn = numpy.sum(nest.GetStatus(nest.GetConnections([neurons[0]], [neurons[1]]), "weight"))
            #print(tmp_stdp_syn[0], end="\t")
            end_sample.append(tmp_stdp_syn)

        print(numpy.mean(start_sample), numpy.mean(end_sample), end="\t")
    print(end="\n", flush=True)
print(end="\n", flush=True)
